/***************************************************************************
 * Copyright 2012 BMW Car IT GmbH
 * Copyright (C) 2012 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Configuration.h"
#include "Log.h"
#include <getopt.h>
#include <stdlib.h>
#include <string.h>
#include <ilm_types.h>

#define MAX_FILTER_KERNEL 9

const char* USAGE_DESCRIPTION =
    "Usage:\t LayerManagerService [options]\n"
    "options:\t\n\n"
    "\t-w: Window Width\t\n"
    "\t-h: Window Height\t\n"
    "\t-d: displayName \t\n"
    "\t-f: loglevel file \t 0 [default] \n\t\t\t\t[0=disabled, 1=error, 2=info, 3=warning, 4=debug]\n"
    "\t-c: loglevel console \t 2 [default] \n\t\t\t\t[0=disabled, 1=error, 2=info, 3=warning, 4=debug]\n"
    "\t-l: loglevel trace \t 4 [default] \n\t\t\t\t[0=disabled, 1=error, 2=info, 3=warning, 4=debug]\n"
    "\t-x: compositing filter level \t 5 [default] \n\t\t\t\t[0=off,1,3,5,7,9]\n"
    "\t-v: show version info\t\n"
    "\nexample: LayerManagerService -w800 -h480 -d:0\n";

Configuration::Configuration(int argc, char** argv)
: mCommandLine("")
, mDisplayWidth(DEFAULT_DISPLAY_WIDTH)
, mDisplayHeight(DEFAULT_DISPLAY_HEIGHT)
, mLogLevelConsole(DEFAULT_LOG_LEVEL_CONSOLE)
, mLogLevelFile(DEFAULT_LOG_LEVEL_FILE)
, mLogLevelTrace(DEFAULT_LOG_LEVEL_TRACE)
, mDisplayName(DEFAULT_DISPLAY_NAME)
, mPluginPath(DEFAULT_PLUGIN_PATH)
, mDisplayColorDepth(DEFAULT_DISPLAY_COLOR_DEPTH)
, mDisplayBufferPerOutput(DEFAULT_DISPLAY_BUFFER_PER_OUTPUT)
, mDisplayDithering(DEFAULT_DISPLAY_DITHERING)
, mFilterKernelSize(DEFAULT_FILTER_KERNEL_SIZE)
, mIgnoreEventOnAlpha0(DEFAULT_IGNORE_EVENT_ON_ALPHA0)
, mpBuildFlags(gBuildFlags)
, mBuildFlagCount(gBuildFlagCount)
, mCustomConfig(false)
, mDisplayIdForInput(DEFAULT_DISP_ID_FOR_INPUT)
, mInputDevConfigPath(DEFAULT_INPUTDEVFILEPATH)
, mPrimaryCSCdisplayID(DEFAULT_PRIMARY_CSC_DISP_ID)
, mSecondaryCSCdisplayID(DEFAULT_SECONDARY_CSC_DISP_ID)
, mRepaintWindowPeriodMS(DEFAULT_REPAINT_WINDOW_PERIOD_MS)
, mEnableMultiSrcBlit(DEFAULT_USE_MULTI_SRC_BLIT)
{
    unsigned int countDpy = 0;

    for (countDpy = 0; countDpy < (unsigned int)ILM_DISPLAY_MAX_NUM; countDpy++)
    {
        mDisplayIds[countDpy] = countDpy;
    }

    processBuildFlags();
    processEnvironmentVariables();
    processCommandLine(argc, argv);
}

std::string Configuration::getDisplayName()
{
    return mDisplayName;
}

int Configuration::getDisplayWidth()
{
    return mDisplayWidth;
}

int Configuration::getDisplayHeight()
{
    return mDisplayHeight;
}

int Configuration::getLogLevelConsole()
{
    return mLogLevelConsole;
}

int Configuration::getLogLevelFile()
{
    return mLogLevelFile;
}

int Configuration::getLogLevelTrace()
{
    return mLogLevelTrace;
}

std::string Configuration::getPluginPath()
{
    return mPluginPath;
}

int Configuration::getDisplayColorDepth()
{
    return mDisplayColorDepth;
}

int Configuration::getDisplayBufferCnt()
{
    return mDisplayBufferPerOutput;
}

bool Configuration::getDisplayDithering()
{
    return mDisplayDithering;
}

int Configuration::getFilterKernelSize()
{
    return mFilterKernelSize;
}

void Configuration::disableFilterBlit()
{
    mFilterKernelSize = 0;
}

bool Configuration::getIgnoreEventOnAlpha0()
{
    return mIgnoreEventOnAlpha0;
}

int Configuration::getDisplayForInput()
{
    return mDisplayIdForInput;
}
int Configuration::getScreenColorDepth(unsigned int screen)
{
	char buff[30];
	const char* var = NULL;
    if (mCustomConfig)
    {
		sprintf(buff, "LM_BUF_DEPTH_SCREEN_%d", screen);
		var = getenv(buff);
		if (var)
		{
			int depth = atoi(var);
			if (depth == 16)
			{
				return  16;
			}
			else if (depth == 32)
			{
				return  32;
			}

		}
    }
    return mDisplayColorDepth;
}

int Configuration::getScreenBufferCnt(unsigned int screen)
{
	char buff[40];
	const char* var = NULL;
    if (mCustomConfig)
    {
		sprintf(buff, "LM_BUF_PER_OUTPUT_SCREEN_%d", screen);
		var = getenv(buff);
		if (var)
		{
			int buf_per_output = atoi(var);
			if (buf_per_output == 2)
			{
				return 2;
			}
			else if (buf_per_output == 3)
			{
				return 3;
			}
		}
    }
	return mDisplayBufferPerOutput;
}

bool Configuration::getScreenDithering(unsigned int screen)
{
	char buff[30];
	const char* var = NULL;
	if (mCustomConfig)
	{
		sprintf(buff, "LM_DITHERING_SCREEN_%d", screen);
		var = getenv(buff);
		if (var)
		{
			if ( (strcmp(var,"OFF")== 0) ||
				(strcmp(var,"off")== 0))
			{
				return false;
			}
			else
			{
				return true;
			}
		}
    }
    return mDisplayDithering;
}

std::string
Configuration::getInputDevConfigPath() {

    return mInputDevConfigPath;
}

void
Configuration::setInputDevConfigPath(std::string devconfigpath)
{
    if (devconfigpath.empty())
    {
        LOG_ERROR("Configuration","provided device input path is empty!");
    }

    LOG_WARNING("Configuration", "new path for input device config: " << devconfigpath);
    mInputDevConfigPath = devconfigpath;
}

void Configuration::logAllSettings()
{
    LOG_INFO("Configuration", "Command Line=" << mCommandLine);
    LOG_INFO("Configuration",
             "Display=" << mDisplayName << ", size=" << mDisplayWidth << ", "
             "x=" << mDisplayHeight);
    LOG_INFO("Configuration", "Plugin path=" << mPluginPath);
    LOG_INFO("Configuration",
             "LM_DISP_BUF_DEPTH=" << mDisplayColorDepth);
    LOG_INFO("Configuration",
             "LM_DISPLAY_DITHERING=" << mDisplayDithering);
    LOG_INFO("Configuration",
             "LM_DISP_BUF_PER_OUTPUT=" << mDisplayBufferPerOutput);
    LOG_INFO("Configuration",
             "LM_IGNORE_EVENT_ON_ALPHA0=" << mIgnoreEventOnAlpha0);
    LOG_INFO("Configuration",
             "LM_DISPLAY_ID_FOR_INPUT=" << mDisplayIdForInput);
    LOG_INFO("Configuration",
             "Log level: console=" << mLogLevelConsole << ", "
             "file=" << mLogLevelFile << ", trace=" << mLogLevelTrace);
    LOG_INFO("Configuration",
              "LM_INPUTDEVCONFIG_PATH = " << mInputDevConfigPath);
    LOG_INFO("Configuration",
              "LM CSC DISPLAY IDS, PRIMARY = " << mPrimaryCSCdisplayID <<
                                ", SECONDARY = "<< mSecondaryCSCdisplayID);

    for (int i = 0; i < mBuildFlagCount; ++i)
    {
        const BuildFlag& flag = mpBuildFlags[i];
        switch (flag.type)
        {
        case DEBUG_FLAG:
            LOG_DEBUG("Configuration", flag.description);
            break;
        case INFO_FLAG:
            LOG_INFO("Configuration", flag.description);
            break;
        }
    }
}

void Configuration::processCommandLine(int argc, char** argv)
{
    while (optind < argc)
    {
        int option = getopt(argc, argv, "w::h::d::?::c::f::v::l::x::");
        switch (option)
        {
        case 'd':
			if (NULL != optarg)
				mDisplayName = optarg;
			break;

		case 'w':
			if ((NULL != optarg) and (isdigit(*optarg)))
			{
				if (0 != atoi(optarg))
					mDisplayWidth = atoi(optarg);
			}
			break;

		case 'h':
			if ((NULL != optarg) and (isdigit(*optarg)))
			{
				if (0 != atoi(optarg))
					mDisplayHeight = atoi(optarg);
			}
			break;

		case 'c':
			if ((NULL != optarg) and (isdigit(*optarg)))
			{
				if (atoi(optarg) < LOG_MAX_LEVEL)
				{
					mLogLevelConsole = (LOG_MODES)atoi(optarg);
				}
			}
			break;

		case 'f':
			if ((NULL != optarg) and (isdigit(*optarg)))
			{
				if (atoi(optarg) < LOG_MAX_LEVEL)
				{
					mLogLevelFile = (LOG_MODES)atoi(optarg);
				}
			}
			break;

		case 'l':
			if ((NULL != optarg) and (isdigit(*optarg)))
			{
				if (atoi(optarg) < LOG_MAX_LEVEL)
				{
					mLogLevelTrace = (LOG_MODES)atoi(optarg);
				}
			}
			break;

        case 'v':
            LOG_INFO("Configuration", "Version=" << ILM_VERSION);
            exit(-1);
            break;
        case 'x':
            if ((NULL != optarg) and (isdigit(*optarg)))
            {
                int arg = atoi(optarg);
                if ((arg <= MAX_FILTER_KERNEL &&
                    (arg % 2 > 0)) || (arg == 0 && optarg[0] == '0'))
                {
                    mFilterKernelSize = arg;
                }
            }
            break;
        case '?':
        default:
            LOG_INFO("Configuration", "Version=" << ILM_VERSION);
            puts(USAGE_DESCRIPTION);
            exit(-1);
        }
    }

    std::stringstream commandLine;
    for (int i = 0; i < argc; ++i)
    {
        commandLine << argv[i] << " ";
    }
    mCommandLine = commandLine.str();
}

void Configuration::processBuildFlags()
{
}

bool Configuration::setDisplayIDmapping(
        unsigned int HDMI, unsigned int LVDS0,
        unsigned  int LVDS1, unsigned  int PARALLEL)
{

    /*check for duplicates */
    if (HDMI == LVDS0 || HDMI == LVDS1 || HDMI == PARALLEL ||
        LVDS0 == LVDS1 || LVDS0 == PARALLEL || LVDS1 == PARALLEL)
    {
        LOG_ERROR("Configuration", "invalid display mapping, duplicated ids!");
        return false;
    }

    if (HDMI > ILM_DISPLAY_UNKNOWN ||
        LVDS0 > ILM_DISPLAY_UNKNOWN ||
        LVDS1 > ILM_DISPLAY_UNKNOWN ||
        PARALLEL > ILM_DISPLAY_UNKNOWN)
    {
        LOG_ERROR("Configuration", "invalid display mapping, ids bigger then: "
                <<ILM_DISPLAY_UNKNOWN);
        return false;
    }

    LOG_WARNING("Configuration", " NEW MAPPING HDMI: " <<HDMI
            <<", LVDS0: "<<LVDS0
            <<", LVDS1: "<<LVDS1
            <<", PARALLEL: "<<PARALLEL);

    mDisplayIds[ILM_DISPLAY_HDMI] = HDMI;
    mDisplayIds[ILM_DISPLAY_LVDS0] = LVDS0;
    mDisplayIds[ILM_DISPLAY_LVDS1] = LVDS1;
    mDisplayIds[ILM_DISPLAY_PARALLEL] = PARALLEL;

    return true;
}

bool Configuration::containsNoDuplicates(unsigned int *pDisplayIdArray, unsigned int length)
{
    bool ret = true;
    unsigned int countDpy = 0;
    unsigned int compareDpy = 0;

    for (countDpy = 0; (countDpy < (unsigned int)length) && ret; countDpy++)
    {
        for (compareDpy = (countDpy + 1); (compareDpy < (unsigned int)length) && ret; compareDpy++)
        {
            if(pDisplayIdArray[countDpy] == pDisplayIdArray[compareDpy])
            {
                ret = false;
            }
        }
    }
    return ret;
}

bool Configuration::containsNoInvalidIds(unsigned int *pDisplayIdArray, unsigned int length)
{
    bool ret = true;
    unsigned int countDpy = 0;
    for (countDpy = 0; (countDpy < (unsigned int)length) && ret; countDpy++)
    {
        if(pDisplayIdArray[countDpy] > ILM_DISPLAY_UNKNOWN)
        {
            ret = false;
        }
    }
    return ret;
}

bool Configuration::setDisplayIds(unsigned int *pDisplayIdArray, unsigned int length)
{
    unsigned int countDpy = 0;
    LOG_WARNING("Configuration", " NEW DISPLAY MAPPING");
    for (countDpy = 0; (countDpy < (unsigned int)length); countDpy++)
    {
        mDisplayIds[countDpy] = pDisplayIdArray[countDpy];
        LOG_WARNING("Configuration", "[" << countDpy << "] = " << mDisplayIds[countDpy]);
    }
    return true;
}

bool Configuration::setDisplayIDmapping(unsigned int *pDisplayIdArray, unsigned int length)
{
    bool ret = false;
    if (length <= ILM_DISPLAY_MAX_NUM)
    {
        ret = containsNoDuplicates(pDisplayIdArray, length);

        if (ret)
        {
            ret = containsNoInvalidIds(pDisplayIdArray, length);

            if (ret)
            {
                ret = setDisplayIds(pDisplayIdArray, length);
            }
            else
            {
                LOG_ERROR("Configuration", "invalid display mapping, ids bigger then: "
                    << ILM_DISPLAY_UNKNOWN);
            }
        }
        else
        {
            LOG_ERROR("Configuration", "invalid display mapping, duplicated ids!");
        }
    }
    else
    {
        LOG_ERROR("Configuration", "Number of display IDs provided " << length
                << " is greater than the number of displays supported: " << ILM_DISPLAY_MAX_NUM);
    }
    return ret;
}

bool Configuration::setCSCValues(unsigned int screenID,ilmCSCProperties* pcscProp,
                                             ilmUSERSpecificCSCvalues* pusercscProp)
{
    ilmUSERSpecificCSCvalues* pusercscPropTemp;
    int loop;

    if(NULL != pcscProp)
        setCSCValues(screenID, pcscProp);

    if(NULL == pusercscProp)
        return false;

    for(loop=0;loop<3;++loop)
    {
        if(pusercscProp->diagnal_matrix[loop] > ILM_CSC_MATRIX_MAX_VALUE)
        {
            LOG_ERROR("Configuration","diagnal matrix exceeds max value range val= "<<pusercscProp->diagnal_matrix[loop]);
            pusercscProp->diagnal_matrix[loop] = ILM_CSC_MATRIX_MAX_VALUE;
        }
    }

    for(loop=0;loop<3;++loop)
    {
        if(pusercscProp->offset_vector[loop] > ILM_CSC_B_OFFSET_SIGNED_FOURTEEN_BIT_VAL)
        {
            LOG_ERROR("Configuration","offset_vector exceeds max value range val= "<<pusercscProp->offset_vector[loop]);
            pusercscProp->offset_vector[loop] = ILM_CSC_B_OFFSET_SIGNED_FOURTEEN_BIT_VAL;
        }
    }

    pusercscPropTemp = new ilmUSERSpecificCSCvalues();

    if(NULL == pusercscPropTemp)
        return false;

    memcpy(pusercscPropTemp->diagnal_matrix,pusercscProp->diagnal_matrix,sizeof(pusercscPropTemp->diagnal_matrix));
    memcpy(pusercscPropTemp->offset_vector,pusercscProp->offset_vector,sizeof(pusercscPropTemp->offset_vector));

    m_userspecificCSCvalues[screenID] = pusercscPropTemp;

    return true;
}

ilmCSCProperties* Configuration::getCSCValues(unsigned int screenID,
                                ilmUSERSpecificCSCvalues* puser_cscvalues)
{
    std::map<unsigned int, ilmCSCProperties*>::const_iterator iter = m_CSCProperties.find(screenID);
    std::map<unsigned int, ilmUSERSpecificCSCvalues*>::const_iterator iter_usercsc = m_userspecificCSCvalues.find(screenID);

    if (iter == m_CSCProperties.end())
    {
        return NULL;
    }

    if (iter_usercsc != m_userspecificCSCvalues.end())
    {
        memcpy(puser_cscvalues,(*iter_usercsc).second,sizeof(ilmUSERSpecificCSCvalues));
    }

    return (*iter).second;

}

bool Configuration::setCSCValues(unsigned int screenID, ilmCSCProperties* pcscProp)
{
    ilmCSCProperties* pcscPropTemp = new ilmCSCProperties();

    if((NULL == pcscProp)||(NULL == pcscPropTemp))
        return false;

    pcscPropTemp->set_to_default = pcscProp->set_to_default;
    pcscPropTemp->hue = pcscProp->hue;
    pcscPropTemp->saturation = pcscProp->saturation;
    pcscPropTemp->brightness = pcscProp->brightness;
    pcscPropTemp->contrast = pcscProp->contrast;
    pcscPropTemp->hue_off = pcscProp->hue_off;
    pcscPropTemp->saturation_off = pcscProp->saturation_off;
    pcscPropTemp->brightness_off = pcscProp->brightness_off;
    pcscPropTemp->type = pcscProp->type;

    m_CSCProperties[screenID] = pcscPropTemp;

    return true;
}

ilmCSCProperties* Configuration::getCSCValues(unsigned int screenID)
{
    std::map<unsigned int, ilmCSCProperties*>::const_iterator iter = m_CSCProperties.find(screenID);

    if (iter == m_CSCProperties.end())
    {
        return NULL;
    }

    return (*iter).second;
}

void Configuration::processEnvironmentVariables()
{
    const char* var = getenv("LM_PLUGIN_PATH");
    if (var)
    {
        mPluginPath = var;
    }
    var = getenv("LM_PER_SCREEN_CONFIG");
    if (var)
    {
        if ( (strcmp(var,"ON")== 0) ||
            (strcmp(var,"on")== 0))
        {
        	mCustomConfig = true;
        }
    }
    var = getenv("LM_DISP_BUF_DEPTH");
    if (var)
    {
        int depth = atoi(var);
        if (depth == 16)
        {
            mDisplayColorDepth = 16;
        }
        else if (depth == 32)
        {
            mDisplayColorDepth = 32;
        }
        else
        {/*default value*/
            mDisplayColorDepth = DEFAULT_DISPLAY_COLOR_DEPTH;
        }
    }
    var = getenv("LM_DISP_BUF_PER_OUTPUT");
    if (var)
    {
        int buf_per_output = atoi(var);
        if (buf_per_output == 2)
        {
            mDisplayBufferPerOutput = 2;
        }
        else if (buf_per_output == 3)
        {
            mDisplayBufferPerOutput = 3;
        }
        else
        {
            mDisplayBufferPerOutput = DEFAULT_DISPLAY_BUFFER_PER_OUTPUT;
        }
    }
    var = getenv("LM_DISPLAY_DITHERING");
    if (var)
    {
        if ( (strcmp(var,"OFF")== 0) ||
            (strcmp(var,"off")== 0))
        {
            mDisplayDithering = false;
        }
        else
        {/*dithering is true per default*/
            mDisplayDithering = DEFAULT_DISPLAY_DITHERING;
        }
    }
    var = getenv("LM_IGNORE_EVENT_ON_ALPHA0");
    if (var)
    {
        if ( (strcmp(var,"ON")== 0) ||
            (strcmp(var,"on")== 0))
        {
            mIgnoreEventOnAlpha0 = true;
        }
        else
        {/*ignore event on alpha0 is false per default*/
            mIgnoreEventOnAlpha0 = DEFAULT_IGNORE_EVENT_ON_ALPHA0;
        }
    }

    var = getenv("LM_DISP_ID_FOR_INPUT");
    if (var)
    {
        int dispid_for_input = atoi(var);
        if ((dispid_for_input < 0) || (dispid_for_input >= ILM_DISPLAY_UNKNOWN))
        {
            mDisplayIdForInput = DEFAULT_DISP_ID_FOR_INPUT;
        }
        else
            mDisplayIdForInput = dispid_for_input;
    }

    var = getenv("LM_INPUTDEVCONFIG_PATH");
    if (var)
    {
        mInputDevConfigPath= var;
    }

    var = getenv("LM_PRIMARY_CSC_DISP_ID");
    if (var)
    {
        int prim_csc_dispid = atoi(var);
        if ((prim_csc_dispid < -1) || (prim_csc_dispid >= ILM_DISPLAY_UNKNOWN))
        {
            mPrimaryCSCdisplayID = DEFAULT_PRIMARY_CSC_DISP_ID;
        }
        else
            mPrimaryCSCdisplayID = prim_csc_dispid;
    }

    var = getenv("LM_SECONDARY_CSC_DISP_ID");
    if (var)
    {
        int second_csc_dispid = atoi(var);
        if ((second_csc_dispid < -1) || (second_csc_dispid >= ILM_DISPLAY_UNKNOWN))
        {
            mSecondaryCSCdisplayID = DEFAULT_SECONDARY_CSC_DISP_ID;
        }
        else
            mSecondaryCSCdisplayID = second_csc_dispid;
    }
    if (mPrimaryCSCdisplayID == mSecondaryCSCdisplayID)
    {
        LOG_ERROR("Configuration", "same display id for primary and secondary csc is not allowed"
                " use default values(prim:"<<DEFAULT_PRIMARY_CSC_DISP_ID<<
                " ,second:"<<DEFAULT_SECONDARY_CSC_DISP_ID<<")");
    }

    var = getenv("LM_REPAINT_WINDOW_PERIOD_MS");
    if (var)
    {
        int repaintWindowPeriod = atoi(var);

        if (repaintWindowPeriod <= 0)
            mRepaintWindowPeriodMS = DEFAULT_REPAINT_WINDOW_PERIOD_MS;
        else
            mRepaintWindowPeriodMS = repaintWindowPeriod;
    }

    var = getenv("LM_MULTI_SRC_BLIT");
    if (var)
    {
        if ((strcmp(var,"ON")== 0) ||
            (strcmp(var,"on")== 0))
        {
            mEnableMultiSrcBlit = 1;
        }
        else if ((strcmp(var,"OFF")== 0) ||
                (strcmp(var,"off")== 0))
        {
            mEnableMultiSrcBlit = 0;
        }
    }
}
